<?php
require_once __DIR__.'/../core/DB.php';
require_once __DIR__.'/../core/FieldManager.php';

// Simple authentication - replace with proper auth later
session_start();
if (!isset($_SESSION['admin_logged_in'])) {
    header("Location: /admin/login.php");
    exit;
}

$db = new DB();
$fieldManager = new FieldManager($db);

// Handle field addition
if (isset($_POST['new_field'])) {
    try {
        $fieldManager->addField('posts', $_POST['field_name'], $_POST['field_type']);
        $message = "Field added successfully!";
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Handle post submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['title'])) {
    try {
        $title = trim($_POST['title']);
        if (empty($title)) throw new Exception("Title cannot be empty");

        // ====== START UPDATED SLUG HANDLING ======
        // Get or generate slug
        $slug = $_POST['custom_slug'] ?? '';
        $slug = empty($slug) ? strtolower(preg_replace('/[^a-z0-9]+/i', '-', $title)) : $slug;

        // Ensure slug is unique
        $counter = 1;
        $originalSlug = $slug;
        while ($db->query("SELECT 1 FROM posts WHERE BINARY slug = ?", [$slug])->fetch()) {
            $slug = $originalSlug . '-' . $counter++;
        }
        // ====== END UPDATED SLUG HANDLING ======

        // Get all non-reserved columns
        $allColumns = $db->query("SHOW COLUMNS FROM posts")->fetchAll();
        $reservedColumns = ['id', 'created_at'];
        $dbColumns = array_column($allColumns, 'Field');

        // Start with required fields
        $columns = ['title', 'content', 'slug'];
        $values = [
            trim($_POST['title']),
            $_POST['content'] ?? '',
            $slug
        ];

        // Add custom fields (excluding reserved and required fields)
        foreach ($dbColumns as $col) {
            if (!in_array($col, $reservedColumns) &&
                !in_array($col, ['title', 'content', 'slug']) &&
                isset($_POST[$col]))
            {
                $columns[] = $col;
                $values[] = $_POST[$col] === '' ? null : $_POST[$col];
            }
        }

        // Build and execute query
        $query = sprintf(
            "INSERT INTO posts (%s) VALUES (%s)",
            implode(', ', $columns),
            implode(', ', array_fill(0, count($values), '?'))
        );

        $db->query($query, $values);
        $success = "Post created successfully!";
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Fetch posts and columns
$posts = $db->query("SELECT * FROM posts")->fetchAll();
$columns = $db->query("SHOW COLUMNS FROM posts")->fetchAll();
$reservedColumns = ['id', 'created_at'];
?>
<!DOCTYPE html>
<html>
<head>
    <title>Admin</title>
    <style>
        body { font-family: Arial; padding: 20px; }
        .field { margin-bottom: 1rem; }
        textarea { width: 100%; min-height: 100px; }
        .slug-field { background: #f0f0f0; padding: 10px; margin: 10px 0; }
    </style>
</head>
<body>
<?php if (isset($message)): ?>
    <div style="color: green"><?= $message ?></div>
<?php endif; ?>
<?php if (isset($error)): ?>
    <div style="color: red"><?= $error ?></div>
<?php endif; ?>

<h1>Post Editor</h1>

<!-- Field Builder -->
<div style="background: #f5f5f5; padding: 1rem; margin-bottom: 2rem;">
    <h3>Add New Field</h3>
    <form method="POST">
        <input type="text" name="field_name" placeholder="Field name (e.g., subtitle)" pattern="[a-z0-9_]+" required>
        <select name="field_type" required>
            <option value="text">Short Text</option>
            <option value="longtext">Long Text</option>
            <option value="number">Number</option>
        </select>
        <button type="submit" name="new_field">Add Field</button>
    </form>
</div>

<!-- Post Form -->
<form method="POST">
    <?php foreach ($columns as $col): ?>
        <?php if (!in_array($col['Field'], $reservedColumns)): ?>
            <div class="field">
                <label><?= ucfirst($col['Field']) ?></label>
                <?php if (strpos($col['Type'], 'varchar') !== false || $col['Type'] === 'text'): ?>
                    <input type="text" name="<?= $col['Field'] ?>" value="">
                <?php elseif ($col['Type'] === 'longtext'): ?>
                    <textarea name="<?= $col['Field'] ?>"></textarea>
                <?php elseif ($col['Type'] === 'int'): ?>
                    <input type="number" name="<?= $col['Field'] ?>">
                <?php endif; ?>
            </div>
        <?php endif; ?>
    <?php endforeach; ?>

    <!-- Custom Slug Field -->
    <div class="field slug-field">
        <label><strong>Custom URL Slug</strong></label>
        <input type="text" name="custom_slug" placeholder="Leave blank to auto-generate">
        <small>Example: 'about-our-company'</small>
    </div>

    <button type="submit">Save Post</button>
</form>

<!-- Existing Posts -->
<!-- Existing Posts -->
<h2>Existing Posts</h2>
<ul>
    <?php foreach ($posts as $post): ?>
        <li>
            <strong><?= htmlspecialchars($post['title'] ?? 'Untitled') ?></strong>
            <small>(Slug: <?= htmlspecialchars($post['slug'] ?? '') ?>)</small>
            <a href="../public/index.php?slug=<?= $post['slug'] ?? '' ?>">View</a>
            <a href="edit-post.php?id=<?= $post['id'] ?>" style="color: orange;">Edit</a>
        </li>
    <?php endforeach; ?>
</ul>
<h2>Field Management</h2>
<form method="POST" action="delete-field.php">
    <select name="field_name" required>
        <?php foreach ($columns as $col): ?>
            <?php if (!in_array($col['Field'], ['id', 'title', 'content', 'slug', 'created_at'])): ?>
                <option value="<?= $col['Field'] ?>">
                    <?= $col['Field'] ?> (<?= $col['Type'] ?>)
                </option>
            <?php endif; ?>
        <?php endforeach; ?>
    </select>
    <button type="submit" name="delete_field">Delete Field</button>
</form>
</body>
</html>